/****************************************************************************
* General description
This code includes the implementation of our method below.
[1] "Fast 2D Complex Gabor Filter With Kernel Decomposition", IEEE Trans. Image Processing, 2018

We also provide the implemenation of two existing approaches for performance comparison.
[2] I. T. Young, L. J. V. Vliet, and M. V. Ginkel, "Recursive Gabor filtering," IEEE Trans. Signal Processing, vol. 50, no. 11, pp. 2798-2805, Nov. 2002.
[3] A. Bernardino and J. Santos-Victor, "Fast IIR isotropic 2-D complex Gabor filters with boundary initialization," IEEE Trans. Image Processing, vol. 15, no. 11, pp. 3338-3348, Nov. 2006.

Additionally, straightforward implementation of original Gabor filter bank is provided here for an objective evaluation of Gabor filtering results.
Please refer to our paper [1] for more details about the objective evaluation.

The functions provided in this code are as below.
[1]: 'ourMethod'
[2]: 'recursiveGabor'
[3]: 'fastIIRIsptropicGabor'
Original Gabor filter: 'straightForwardGabor'


* Requirement
To run this code, OpenCV should be installed in your computer.


* Citation
Please cite the following paper if you use this code.

@article{DBLP:journals/tip/KimUM18,
author    = {Jaeyoon Kim and
Suhyuk Um and
Dongbo Min},
title     = {Fast 2D Complex Gabor Filter With Kernel Decomposition},
journal   = {{IEEE} Trans. Image Processing},
volume    = {27},
number    = {4},
pages     = {1713--1722},
year      = {2018},
url       = {https://doi.org/10.1109/TIP.2017.2783621},
doi       = {10.1109/TIP.2017.2783621},
timestamp = {Thu, 15 Feb 2018 17:56:05 +0100},
biburl    = {https://dblp.org/rec/bib/journals/tip/KimUM18},
bibsource = {dblp computer science bibliography, https://dblp.org}
}

*****************************************************************************/


#include <opencv\highgui.h>
#include <opencv2\highgui.hpp>
#include <opencv\cxcore.h>
#include <opencv2\opencv.hpp>
#include <math.h>
#include <stdio.h>
#include <iostream>
#include <vector>
#include <fstream>
#include <time.h> 
#include <string.h>
#include <windows.h>
using namespace cv;
using namespace std;


float atWrapperInX(Mat* m, int x, int y);
float atWrapperInY(Mat* m, int x, int y);
float atWrapperInAll(Mat* m, int x, int y);




Mat* makeSignalCosine(Mat* inputMat, float freq);
Mat* makeSignalSine(Mat* inputMat, float freq);
Mat* makeSignalCosine2(Mat* inputMat, float freq);
Mat* makeSignalSine2(Mat* inputMat, float freq);
void ourImshow(Mat inputR, Mat inputI, int u, int v);
void ourImwrite(Mat inputR, Mat inputI, string n, int u, int v);

Mat* makeSignalCosineInXYDirection(Mat* inputMat, float omegaX, float omegaY);
Mat* makeSignalSineInXYDirection(Mat* inputMat, float omegaX, float omegaY);


void fastIIRIsptropicGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag);
void recursiveGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag);
void ourMethod(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag, Mat* symmetricReal, Mat* symmetricImag);

void straightForwardGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag);

double getnoise(Mat* groundtruthReal, Mat* groundtruthImag, Mat* resultReal, Mat* resultImag, int x, int y);
double getSignal(Mat* resultReal, Mat* resultImag, int x, int y);
void mat2ZerosMat(Mat* resultReal, Mat* resultImag);
void memFreeFloat4(float ****p);
float**** memAllocFloat4(int u, int v, int x, int y);
void memFreeFloat2(float **p);
float** memAllocFloat2(int x, int y);
void memFreeFloat3(float ***p);
float *** memAllocFloat3(int x, int y, int z);
float* memAllocFloat1(int x);
void memFreeFloat1(float *p);
void SEROfThetaVariation(Mat inputs[], int numOfImage, int numOfTheta);
void SEROfLambdaVariation(Mat inputs[], int numOfImage, int numOfLambda);
void aerials();
void misc();
void sequences();
void textures();
const float pi = 3.141592;



int main() {

	aerials();
	misc();
	sequences();
	textures();

	waitKey(0);
}


void aerials() {
	Mat inputs[38];
	Mat originMat = imread("aerials/2.1.01.tiff", 0);
	originMat.convertTo(inputs[0], CV_32F);
	originMat = imread("aerials/2.1.02.tiff", 0);
	originMat.convertTo(inputs[1], CV_32F);
	originMat = imread("aerials/2.1.03.tiff", 0);
	originMat.convertTo(inputs[2], CV_32F);
	originMat = imread("aerials/2.1.04.tiff", 0);
	originMat.convertTo(inputs[3], CV_32F);
	originMat = imread("aerials/2.1.05.tiff", 0);
	originMat.convertTo(inputs[4], CV_32F);
	originMat = imread("aerials/2.1.06.tiff", 0);
	originMat.convertTo(inputs[5], CV_32F);
	originMat = imread("aerials/2.1.07.tiff", 0);
	originMat.convertTo(inputs[6], CV_32F);
	originMat = imread("aerials/2.1.08.tiff", 0);
	originMat.convertTo(inputs[7], CV_32F);
	originMat = imread("aerials/2.1.09.tiff", 0);
	originMat.convertTo(inputs[8], CV_32F);
	originMat = imread("aerials/2.1.10.tiff", 0);
	originMat.convertTo(inputs[9], CV_32F);
	originMat = imread("aerials/2.1.11.tiff", 0);
	originMat.convertTo(inputs[10], CV_32F);
	originMat = imread("aerials/2.1.12.tiff", 0);
	originMat.convertTo(inputs[11], CV_32F);
	originMat = imread("aerials/2.2.01.tiff", 0);
	originMat.convertTo(inputs[12], CV_32F);
	originMat = imread("aerials/2.2.02.tiff", 0);
	originMat.convertTo(inputs[13], CV_32F);
	originMat = imread("aerials/2.2.03.tiff", 0);
	originMat.convertTo(inputs[14], CV_32F);
	originMat = imread("aerials/2.2.04.tiff", 0);
	originMat.convertTo(inputs[15], CV_32F);
	originMat = imread("aerials/2.2.05.tiff", 0);
	originMat.convertTo(inputs[16], CV_32F);
	originMat = imread("aerials/2.2.06.tiff", 0);
	originMat.convertTo(inputs[17], CV_32F);
	originMat = imread("aerials/2.2.07.tiff", 0);
	originMat.convertTo(inputs[18], CV_32F);
	originMat = imread("aerials/2.2.08.tiff", 0);
	originMat.convertTo(inputs[19], CV_32F);
	originMat = imread("aerials/2.2.09.tiff", 0);
	originMat.convertTo(inputs[20], CV_32F);
	originMat = imread("aerials/2.2.10.tiff", 0);
	originMat.convertTo(inputs[21], CV_32F);
	originMat = imread("aerials/2.2.11.tiff", 0);
	originMat.convertTo(inputs[22], CV_32F);
	originMat = imread("aerials/2.2.12.tiff", 0);
	originMat.convertTo(inputs[23], CV_32F);
	originMat = imread("aerials/2.2.13.tiff", 0);
	originMat.convertTo(inputs[24], CV_32F);
	originMat = imread("aerials/2.2.14.tiff", 0);
	originMat.convertTo(inputs[25], CV_32F);
	originMat = imread("aerials/2.2.15.tiff", 0);
	originMat.convertTo(inputs[26], CV_32F);
	originMat = imread("aerials/2.2.16.tiff", 0);
	originMat.convertTo(inputs[27], CV_32F);
	originMat = imread("aerials/2.2.17.tiff", 0);
	originMat.convertTo(inputs[28], CV_32F);
	originMat = imread("aerials/2.2.18.tiff", 0);
	originMat.convertTo(inputs[29], CV_32F);
	originMat = imread("aerials/2.2.19.tiff", 0);
	originMat.convertTo(inputs[30], CV_32F);
	originMat = imread("aerials/2.2.20.tiff", 0);
	originMat.convertTo(inputs[31], CV_32F);
	originMat = imread("aerials/2.2.21.tiff", 0);
	originMat.convertTo(inputs[32], CV_32F);
	originMat = imread("aerials/2.2.22.tiff", 0);
	originMat.convertTo(inputs[33], CV_32F);
	originMat = imread("aerials/2.2.23.tiff", 0);
	originMat.convertTo(inputs[34], CV_32F);
	originMat = imread("aerials/2.2.24.tiff", 0);
	originMat.convertTo(inputs[35], CV_32F);
	originMat = imread("aerials/3.2.25.tiff", 0);
	originMat.convertTo(inputs[36], CV_32F);
	originMat = imread("aerials/wash-ir.tiff", 0);
	originMat.convertTo(inputs[37], CV_32F);

	cout << "-----------aerial------------ " << endl;
	SEROfThetaVariation(inputs, 38, 10);
	SEROfLambdaVariation(inputs, 38, 10);
}
void misc() {
	Mat inputs[42];
	Mat originMat = imread("misc/4.1.01.tiff", 0);
	originMat.convertTo(inputs[0], CV_32F);
	originMat = imread("misc/4.1.02.tiff", 0);
	originMat.convertTo(inputs[1], CV_32F);
	originMat = imread("misc/4.1.03.tiff", 0);
	originMat.convertTo(inputs[2], CV_32F);
	originMat = imread("misc/4.1.04.tiff", 0);
	originMat.convertTo(inputs[3], CV_32F);
	originMat = imread("misc/4.1.05.tiff", 0);
	originMat.convertTo(inputs[4], CV_32F);
	originMat = imread("misc/4.1.06.tiff", 0);
	originMat.convertTo(inputs[5], CV_32F);
	originMat = imread("misc/4.1.07.tiff", 0);
	originMat.convertTo(inputs[6], CV_32F);
	originMat = imread("misc/4.1.08.tiff", 0);
	originMat.convertTo(inputs[7], CV_32F);
	originMat = imread("misc/4.2.01.tiff", 0);
	originMat.convertTo(inputs[8], CV_32F);
	originMat = imread("misc/4.2.02.tiff", 0);
	originMat.convertTo(inputs[9], CV_32F);
	originMat = imread("misc/4.2.03.tiff", 0);
	originMat.convertTo(inputs[10], CV_32F);
	originMat = imread("misc/4.2.04.tiff", 0);
	originMat.convertTo(inputs[11], CV_32F);
	originMat = imread("misc/4.2.05.tiff", 0);
	originMat.convertTo(inputs[12], CV_32F);
	originMat = imread("misc/4.2.06.tiff", 0);
	originMat.convertTo(inputs[13], CV_32F);
	originMat = imread("misc/4.2.07.tiff", 0);
	originMat.convertTo(inputs[14], CV_32F);
	originMat = imread("misc/5.1.09.tiff", 0);
	originMat.convertTo(inputs[15], CV_32F);
	originMat = imread("misc/5.1.10.tiff", 0);
	originMat.convertTo(inputs[16], CV_32F);
	originMat = imread("misc/5.1.11.tiff", 0);
	originMat.convertTo(inputs[17], CV_32F);
	originMat = imread("misc/5.1.12.tiff", 0);
	originMat.convertTo(inputs[18], CV_32F);
	originMat = imread("misc/5.1.13.tiff", 0);
	originMat.convertTo(inputs[19], CV_32F);
	originMat = imread("misc/5.1.14.tiff", 0);
	originMat.convertTo(inputs[20], CV_32F);
	originMat = imread("misc/5.2.08.tiff", 0);
	originMat.convertTo(inputs[21], CV_32F);
	originMat = imread("misc/5.2.09.tiff", 0);
	originMat.convertTo(inputs[22], CV_32F);
	originMat = imread("misc/5.2.10.tiff", 0);
	originMat.convertTo(inputs[23], CV_32F);
	originMat = imread("misc/5.3.01.tiff", 0);
	originMat.convertTo(inputs[24], CV_32F);
	originMat = imread("misc/5.3.02.tiff", 0);
	originMat.convertTo(inputs[25], CV_32F);
	originMat = imread("misc/7.1.01.tiff", 0);
	originMat.convertTo(inputs[26], CV_32F);
	originMat = imread("misc/7.1.02.tiff", 0);
	originMat.convertTo(inputs[27], CV_32F);
	originMat = imread("misc/7.1.03.tiff", 0);
	originMat.convertTo(inputs[28], CV_32F);
	originMat = imread("misc/7.1.04.tiff", 0);
	originMat.convertTo(inputs[29], CV_32F);
	originMat = imread("misc/7.1.05.tiff", 0);
	originMat.convertTo(inputs[30], CV_32F);
	originMat = imread("misc/7.1.06.tiff", 0);
	originMat.convertTo(inputs[31], CV_32F);
	originMat = imread("misc/7.1.07.tiff", 0);
	originMat.convertTo(inputs[32], CV_32F);
	originMat = imread("misc/7.1.08.tiff", 0);
	originMat.convertTo(inputs[33], CV_32F);
	originMat = imread("misc/7.1.09.tiff", 0);
	originMat.convertTo(inputs[34], CV_32F);
	originMat = imread("misc/7.1.10.tiff", 0);
	originMat.convertTo(inputs[35], CV_32F);
	originMat = imread("misc/7.2.01.tiff", 0);
	originMat.convertTo(inputs[36], CV_32F);
	originMat = imread("misc/boat.512.tiff", 0);
	originMat.convertTo(inputs[37], CV_32F);
	originMat = imread("misc/elaine.512.tiff", 0);
	originMat.convertTo(inputs[36], CV_32F);
	originMat = imread("misc/gray21.512.tiff", 0);
	originMat.convertTo(inputs[37], CV_32F);
	originMat = imread("misc/house.tiff", 0);
	originMat.convertTo(inputs[38], CV_32F);
	originMat = imread("misc/numbers.512.tiff", 0);
	originMat.convertTo(inputs[39], CV_32F);
	originMat = imread("misc/ruler.512.tiff", 0);
	originMat.convertTo(inputs[40], CV_32F);
	originMat = imread("misc/testpat.1k.tiff", 0);
	originMat.convertTo(inputs[41], CV_32F);

	cout << "-----------misc------------ " << endl;
	SEROfThetaVariation(inputs, 42, 10);
	SEROfLambdaVariation(inputs, 42, 10);
}
void sequences() {
	Mat inputs[69];
	Mat originMat = imread("sequences/6.1.01.tiff", 0);
	originMat.convertTo(inputs[0], CV_32F);
	originMat = imread("sequences/6.1.02.tiff", 0);
	originMat.convertTo(inputs[1], CV_32F);
	originMat = imread("sequences/6.1.03.tiff", 0);
	originMat.convertTo(inputs[2], CV_32F);
	originMat = imread("sequences/6.1.04.tiff", 0);
	originMat.convertTo(inputs[3], CV_32F);
	originMat = imread("sequences/6.1.05.tiff", 0);
	originMat.convertTo(inputs[4], CV_32F);
	originMat = imread("sequences/6.1.06.tiff", 0);
	originMat.convertTo(inputs[5], CV_32F);
	originMat = imread("sequences/6.1.07.tiff", 0);
	originMat.convertTo(inputs[6], CV_32F);
	originMat = imread("sequences/6.1.08.tiff", 0);
	originMat.convertTo(inputs[7], CV_32F);
	originMat = imread("sequences/6.1.09.tiff", 0);
	originMat.convertTo(inputs[8], CV_32F);
	originMat = imread("sequences/6.1.10.tiff", 0);
	originMat.convertTo(inputs[9], CV_32F);
	originMat = imread("sequences/6.1.11.tiff", 0);
	originMat.convertTo(inputs[10], CV_32F);
	originMat = imread("sequences/6.1.12.tiff", 0);
	originMat.convertTo(inputs[11], CV_32F);
	originMat = imread("sequences/6.1.13.tiff", 0);
	originMat.convertTo(inputs[12], CV_32F);
	originMat = imread("sequences/6.1.14.tiff", 0);
	originMat.convertTo(inputs[13], CV_32F);
	originMat = imread("sequences/6.1.15.tiff", 0);
	originMat.convertTo(inputs[14], CV_32F);
	originMat = imread("sequences/6.1.16.tiff", 0);
	originMat.convertTo(inputs[15], CV_32F);
	originMat = imread("sequences/6.2.01.tiff", 0);
	originMat.convertTo(inputs[16], CV_32F);
	originMat = imread("sequences/6.2.02.tiff", 0);
	originMat.convertTo(inputs[17], CV_32F);
	originMat = imread("sequences/6.2.03.tiff", 0);
	originMat.convertTo(inputs[18], CV_32F);
	originMat = imread("sequences/6.2.04.tiff", 0);
	originMat.convertTo(inputs[19], CV_32F);
	originMat = imread("sequences/6.2.05.tiff", 0);
	originMat.convertTo(inputs[20], CV_32F);
	originMat = imread("sequences/6.2.06.tiff", 0);
	originMat.convertTo(inputs[21], CV_32F);
	originMat = imread("sequences/6.2.07.tiff", 0);
	originMat.convertTo(inputs[22], CV_32F);
	originMat = imread("sequences/6.2.08.tiff", 0);
	originMat.convertTo(inputs[23], CV_32F);
	originMat = imread("sequences/6.2.09.tiff", 0);
	originMat.convertTo(inputs[24], CV_32F);
	originMat = imread("sequences/6.2.10.tiff", 0);
	originMat.convertTo(inputs[25], CV_32F);
	originMat = imread("sequences/6.2.11.tiff", 0);
	originMat.convertTo(inputs[26], CV_32F);
	originMat = imread("sequences/6.2.12.tiff", 0);
	originMat.convertTo(inputs[27], CV_32F);
	originMat = imread("sequences/6.2.13.tiff", 0);
	originMat.convertTo(inputs[28], CV_32F);
	originMat = imread("sequences/6.2.14.tiff", 0);
	originMat.convertTo(inputs[29], CV_32F);
	originMat = imread("sequences/6.2.15.tiff", 0);
	originMat.convertTo(inputs[30], CV_32F);
	originMat = imread("sequences/6.2.16.tiff", 0);
	originMat.convertTo(inputs[31], CV_32F);
	originMat = imread("sequences/6.2.17.tiff", 0);
	originMat.convertTo(inputs[32], CV_32F);
	originMat = imread("sequences/6.2.18.tiff", 0);
	originMat.convertTo(inputs[33], CV_32F);
	originMat = imread("sequences/6.2.19.tiff", 0);
	originMat.convertTo(inputs[34], CV_32F);
	originMat = imread("sequences/6.2.20.tiff", 0);
	originMat.convertTo(inputs[35], CV_32F);
	originMat = imread("sequences/6.2.21.tiff", 0);
	originMat.convertTo(inputs[36], CV_32F);
	originMat = imread("sequences/6.2.22.tiff", 0);
	originMat.convertTo(inputs[37], CV_32F);
	originMat = imread("sequences/6.2.23.tiff", 0);
	originMat.convertTo(inputs[38], CV_32F);
	originMat = imread("sequences/6.2.24.tiff", 0);
	originMat.convertTo(inputs[39], CV_32F);
	originMat = imread("sequences/6.2.25.tiff", 0);
	originMat.convertTo(inputs[40], CV_32F);
	originMat = imread("sequences/6.2.26.tiff", 0);
	originMat.convertTo(inputs[41], CV_32F);
	originMat = imread("sequences/6.2.27.tiff", 0);
	originMat.convertTo(inputs[42], CV_32F);
	originMat = imread("sequences/6.2.28.tiff", 0);
	originMat.convertTo(inputs[43], CV_32F);
	originMat = imread("sequences/6.2.29.tiff", 0);
	originMat.convertTo(inputs[44], CV_32F);
	originMat = imread("sequences/6.2.30.tiff", 0);
	originMat.convertTo(inputs[45], CV_32F);
	originMat = imread("sequences/6.2.31.tiff", 0);
	originMat.convertTo(inputs[46], CV_32F);
	originMat = imread("sequences/6.2.32.tiff", 0);
	originMat.convertTo(inputs[47], CV_32F);
	originMat = imread("sequences/6.3.01.tiff", 0);
	originMat.convertTo(inputs[48], CV_32F);
	originMat = imread("sequences/6.3.02.tiff", 0);
	originMat.convertTo(inputs[49], CV_32F);
	originMat = imread("sequences/6.3.03.tiff", 0);
	originMat.convertTo(inputs[50], CV_32F);
	originMat = imread("sequences/6.3.04.tiff", 0);
	originMat.convertTo(inputs[51], CV_32F);
	originMat = imread("sequences/6.3.05.tiff", 0);
	originMat.convertTo(inputs[52], CV_32F);
	originMat = imread("sequences/6.3.06.tiff", 0);
	originMat.convertTo(inputs[53], CV_32F);
	originMat = imread("sequences/6.3.07.tiff", 0);
	originMat.convertTo(inputs[54], CV_32F);
	originMat = imread("sequences/6.3.08.tiff", 0);
	originMat.convertTo(inputs[55], CV_32F);
	originMat = imread("sequences/6.3.09.tiff", 0);
	originMat.convertTo(inputs[56], CV_32F);
	originMat = imread("sequences/6.3.10.tiff", 0);
	originMat.convertTo(inputs[57], CV_32F);
	originMat = imread("sequences/6.3.11.tiff", 0);
	originMat.convertTo(inputs[58], CV_32F);
	originMat = imread("sequences/motion01.512.tiff", 0);
	originMat.convertTo(inputs[59], CV_32F);
	originMat = imread("sequences/motion02.512.tiff", 0);
	originMat.convertTo(inputs[60], CV_32F);
	originMat = imread("sequences/motion03.512.tiff", 0);
	originMat.convertTo(inputs[61], CV_32F);
	originMat = imread("sequences/motion04.512.tiff", 0);
	originMat.convertTo(inputs[62], CV_32F);
	originMat = imread("sequences/motion05.512.tiff", 0);
	originMat.convertTo(inputs[63], CV_32F);
	originMat = imread("sequences/motion06.512.tiff", 0);
	originMat.convertTo(inputs[64], CV_32F);
	originMat = imread("sequences/motion07.512.tiff", 0);
	originMat.convertTo(inputs[65], CV_32F);
	originMat = imread("sequences/motion08.512.tiff", 0);
	originMat.convertTo(inputs[66], CV_32F);
	originMat = imread("sequences/motion09.512.tiff", 0);
	originMat.convertTo(inputs[67], CV_32F);
	originMat = imread("sequences/motion10.512.tiff", 0);
	originMat.convertTo(inputs[68], CV_32F);


	cout << "-----------sequences------------ " << endl;
	SEROfThetaVariation(inputs, 69, 10);
	SEROfLambdaVariation(inputs, 69, 10);
}
void textures() {
	Mat inputs[64];
	Mat originMat = imread("textures/1.1.01.tiff", 0);
	originMat.convertTo(inputs[0], CV_32F);
	originMat = imread("textures/1.1.02.tiff", 0);
	originMat.convertTo(inputs[1], CV_32F);
	originMat = imread("textures/1.1.03.tiff", 0);
	originMat.convertTo(inputs[2], CV_32F);
	originMat = imread("textures/1.1.04.tiff", 0);
	originMat.convertTo(inputs[3], CV_32F);
	originMat = imread("textures/1.1.05.tiff", 0);
	originMat.convertTo(inputs[4], CV_32F);
	originMat = imread("textures/1.1.06.tiff", 0);
	originMat.convertTo(inputs[5], CV_32F);
	originMat = imread("textures/1.1.07.tiff", 0);
	originMat.convertTo(inputs[6], CV_32F);
	originMat = imread("textures/1.1.08.tiff", 0);
	originMat.convertTo(inputs[7], CV_32F);
	originMat = imread("textures/1.1.09.tiff", 0);
	originMat.convertTo(inputs[8], CV_32F);
	originMat = imread("textures/1.1.10.tiff", 0);
	originMat.convertTo(inputs[9], CV_32F);
	originMat = imread("textures/1.1.11.tiff", 0);
	originMat.convertTo(inputs[10], CV_32F);
	originMat = imread("textures/1.1.12.tiff", 0);
	originMat.convertTo(inputs[11], CV_32F);
	originMat = imread("textures/1.1.13.tiff", 0);
	originMat.convertTo(inputs[12], CV_32F);
	originMat = imread("textures/1.2.01.tiff", 0);
	originMat.convertTo(inputs[13], CV_32F);
	originMat = imread("textures/1.2.02.tiff", 0);
	originMat.convertTo(inputs[14], CV_32F);
	originMat = imread("textures/1.2.03.tiff", 0);
	originMat.convertTo(inputs[15], CV_32F);
	originMat = imread("textures/1.2.04.tiff", 0);
	originMat.convertTo(inputs[16], CV_32F);
	originMat = imread("textures/1.2.05.tiff", 0);
	originMat.convertTo(inputs[17], CV_32F);
	originMat = imread("textures/1.2.06.tiff", 0);
	originMat.convertTo(inputs[18], CV_32F);
	originMat = imread("textures/1.2.07.tiff", 0);
	originMat.convertTo(inputs[19], CV_32F);
	originMat = imread("textures/1.2.08.tiff", 0);
	originMat.convertTo(inputs[20], CV_32F);
	originMat = imread("textures/1.2.09.tiff", 0);
	originMat.convertTo(inputs[21], CV_32F);
	originMat = imread("textures/1.2.10.tiff", 0);
	originMat.convertTo(inputs[22], CV_32F);
	originMat = imread("textures/1.2.11.tiff", 0);
	originMat.convertTo(inputs[23], CV_32F);
	originMat = imread("textures/1.2.12.tiff", 0);
	originMat.convertTo(inputs[24], CV_32F);
	originMat = imread("textures/1.2.13.tiff", 0);
	originMat.convertTo(inputs[25], CV_32F);


	originMat = imread("textures/1.3.01.tiff", 0);
	originMat.convertTo(inputs[26], CV_32F);
	originMat = imread("textures/1.3.02.tiff", 0);
	originMat.convertTo(inputs[27], CV_32F);
	originMat = imread("textures/1.3.03.tiff", 0);
	originMat.convertTo(inputs[28], CV_32F);
	originMat = imread("textures/1.3.04.tiff", 0);
	originMat.convertTo(inputs[29], CV_32F);
	originMat = imread("textures/1.3.05.tiff", 0);
	originMat.convertTo(inputs[30], CV_32F);
	originMat = imread("textures/1.3.06.tiff", 0);
	originMat.convertTo(inputs[31], CV_32F);
	originMat = imread("textures/1.3.07.tiff", 0);
	originMat.convertTo(inputs[32], CV_32F);
	originMat = imread("textures/1.3.08.tiff", 0);
	originMat.convertTo(inputs[33], CV_32F);
	originMat = imread("textures/1.3.09.tiff", 0);
	originMat.convertTo(inputs[34], CV_32F);
	originMat = imread("textures/1.3.10.tiff", 0);
	originMat.convertTo(inputs[35], CV_32F);
	originMat = imread("textures/1.3.11.tiff", 0);
	originMat.convertTo(inputs[36], CV_32F);
	originMat = imread("textures/1.3.12.tiff", 0);
	originMat.convertTo(inputs[37], CV_32F);
	originMat = imread("textures/1.3.13.tiff", 0);
	originMat.convertTo(inputs[38], CV_32F);
	originMat = imread("textures/1.4.01.tiff", 0);
	originMat.convertTo(inputs[39], CV_32F);
	originMat = imread("textures/1.4.02.tiff", 0);
	originMat.convertTo(inputs[40], CV_32F);
	originMat = imread("textures/1.4.03.tiff", 0);
	originMat.convertTo(inputs[41], CV_32F);
	originMat = imread("textures/1.4.04.tiff", 0);
	originMat.convertTo(inputs[42], CV_32F);
	originMat = imread("textures/1.4.05.tiff", 0);
	originMat.convertTo(inputs[43], CV_32F);
	originMat = imread("textures/1.4.06.tiff", 0);
	originMat.convertTo(inputs[44], CV_32F);
	originMat = imread("textures/1.4.07.tiff", 0);
	originMat.convertTo(inputs[45], CV_32F);
	originMat = imread("textures/1.4.08.tiff", 0);
	originMat.convertTo(inputs[46], CV_32F);
	originMat = imread("textures/1.4.09.tiff", 0);
	originMat.convertTo(inputs[47], CV_32F);
	originMat = imread("textures/1.4.10.tiff", 0);
	originMat.convertTo(inputs[48], CV_32F);
	originMat = imread("textures/1.4.11.tiff", 0);
	originMat.convertTo(inputs[49], CV_32F);
	originMat = imread("textures/1.4.12.tiff", 0);
	originMat.convertTo(inputs[50], CV_32F);
	originMat = imread("textures/1.5.01.tiff", 0);
	originMat.convertTo(inputs[51], CV_32F);
	originMat = imread("textures/1.5.02.tiff", 0);
	originMat.convertTo(inputs[52], CV_32F);
	originMat = imread("textures/1.5.03.tiff", 0);
	originMat.convertTo(inputs[53], CV_32F);
	originMat = imread("textures/1.5.04.tiff", 0);
	originMat.convertTo(inputs[54], CV_32F);
	originMat = imread("textures/1.5.05.tiff", 0);
	originMat.convertTo(inputs[55], CV_32F);
	originMat = imread("textures/1.5.06.tiff", 0);
	originMat.convertTo(inputs[56], CV_32F);
	originMat = imread("textures/1.5.07.tiff", 0);
	originMat.convertTo(inputs[57], CV_32F);
	originMat = imread("textures/texmos1.p512.tiff", 0);
	originMat.convertTo(inputs[58], CV_32F);
	originMat = imread("textures/texmos2.p512.tiff", 0);
	originMat.convertTo(inputs[59], CV_32F);
	originMat = imread("textures/texmos2.s512.tiff", 0);
	originMat.convertTo(inputs[60], CV_32F);
	originMat = imread("textures/texmos3.p512.tiff", 0);
	originMat.convertTo(inputs[61], CV_32F);
	originMat = imread("textures/texmos3.s512.tiff", 0);
	originMat.convertTo(inputs[62], CV_32F);
	originMat = imread("textures/texmos3b.p512.tiff", 0);
	originMat.convertTo(inputs[63], CV_32F);



	cout << "-----------textures------------ " << endl;
	SEROfThetaVariation(inputs, 64, 10);
	SEROfLambdaVariation(inputs, 64, 10);

}
void SEROfThetaVariation(Mat inputs[], int numOfImage, int numOfTheta) {

	int ksize;
	float sigma;
	float theta;
	float lambda;
	Mat inputMat;

	for (int n = 1; n < numOfTheta; n++) {
		theta = pi*n / numOfTheta;
		sigma = 3;
		ksize = 6 * sigma;
		lambda = sigma / pi;
		double SER = 0;
		double SER2 = 0;
		double SER3 = 0;

		for (int image = 0; image < numOfImage; image++) {
			inputMat = inputs[image];

			Mat* resultReal = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultImag = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultReal2 = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultImag2 = new Mat(inputMat.size(), CV_32FC1);
			Mat* groundtruthReal = new Mat(inputMat.size(), CV_32FC1);
			Mat* groundtruthImag = new Mat(inputMat.size(), CV_32FC1);
			double MAX = 255;
			double MSE = 0;
			double signal = 0;
			double noise = 0;

			mat2ZerosMat(groundtruthReal, groundtruthImag);
			straightForwardGabor(&inputMat, ksize, sigma, lambda, theta, groundtruthReal, groundtruthImag);

			ourMethod(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag, resultReal2, resultImag2);

			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {

					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}


			SER += 10 * log10(noise / signal);



			mat2ZerosMat(resultReal, resultImag);
			MSE = 0;
			signal = 0;
			noise = 0;
			fastIIRIsptropicGabor(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag);

			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {

					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}


			SER2 += 10 * log10(noise / signal);

			mat2ZerosMat(resultReal, resultImag);
			MSE = 0;
			signal = 0;
			noise = 0;
			recursiveGabor(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag);

			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {
					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}



			SER3 += 10 * log10(noise / signal);

			delete resultReal;
			delete resultImag;
			delete resultReal2;
			delete resultImag2;
			delete groundtruthReal;
			delete groundtruthImag;
		}
		SER /= -numOfImage;
		SER2 /= -numOfImage;
		SER3 /= -numOfImage;

		cout << "theta : pi" << n << "/" << numOfTheta << ", SER1 : " << SER << ", SER2 : " << SER2 << ", SER3 : " << SER3 << endl;
	}
}
void SEROfLambdaVariation(Mat inputs[], int numOfImage, int numOfLambda) {
	int ksize;
	float sigma;
	float theta;
	float lambda;
	Mat inputMat;

	for (int n = 1; n < numOfLambda; n++) {
		theta = pi / 3;
		sigma = n*0.5 + 1;
		lambda = sigma / pi;

		ksize = 6 * sigma;

		double SER = 0;
		double SER2 = 0;
		double SER3 = 0;

		for (int image = 0; image < numOfImage; image++) {
			inputMat = inputs[image];

			Mat* resultReal = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultImag = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultReal2 = new Mat(inputMat.size(), CV_32FC1);
			Mat* resultImag2 = new Mat(inputMat.size(), CV_32FC1);
			Mat* groundtruthReal = new Mat(inputMat.size(), CV_32FC1);
			Mat* groundtruthImag = new Mat(inputMat.size(), CV_32FC1);
			double MAX = 255;
			double MSE = 0;
			double signal = 0;
			double noise = 0;

			mat2ZerosMat(groundtruthReal, groundtruthImag);
			straightForwardGabor(&inputMat, ksize, sigma, lambda, theta, groundtruthReal, groundtruthImag);



			ourMethod(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag, resultReal2, resultImag2);

			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {

					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}


			SER += 10 * log10(noise / signal);



			mat2ZerosMat(resultReal, resultImag);
			MSE = 0;
			signal = 0;
			noise = 0;

			fastIIRIsptropicGabor(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag);

			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {

					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}



			SER2 += 10 * log10(noise / signal);

			mat2ZerosMat(resultReal, resultImag);
			MSE = 0;
			signal = 0;
			noise = 0;
			//MAX = -99999;
			recursiveGabor(&inputMat, ksize, sigma, lambda, theta, resultReal, resultImag);
			//	ourImwrite(*resultReal, *resultImag, "recursive", 1, 1);
			for (int x = ksize; x < inputMat.size().width - ksize; x++)
				for (int y = ksize; y < inputMat.size().height - ksize; y++) {
					noise += getnoise(groundtruthReal, groundtruthImag, resultReal, resultImag, x, y);
					signal += getSignal(resultReal, resultImag, x, y);

				}



			SER3 += 10 * log10(noise / signal);

			delete resultReal;
			delete resultImag;
			delete resultReal2;
			delete resultImag2;
			delete groundtruthReal;
			delete groundtruthImag;
		}
		SER /= -numOfImage;
		SER2 /= -numOfImage;
		SER3 /= -numOfImage;


		cout << "lambda : " << lambda << ", SER1 : " << SER << ", SER2 : " << SER2 << ", SER3 : " << SER3 << endl;
	}
}
void mat2ZerosMat(Mat* resultReal, Mat* resultImag) {
	for (int y = 0; y<resultReal->size().height; y++)
		for (int x = 0; x < resultReal->size().width; x++) {
			resultReal->at<float>(y, x) = 0;
			resultImag->at<float>(y, x) = 0;
		}
}

double getnoise(Mat* groundtruthReal, Mat* groundtruthImag, Mat* resultReal, Mat* resultImag, int x, int y) {
	return pow(-groundtruthReal->at<float>(y, x) + resultReal->at<float>(y, x), 2);
}

double getSignal(Mat* resultReal, Mat* resultImag, int x, int y) {
	return pow(resultReal->at<float>(y, x), 2);
}

void straightForwardGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag) {
	int khalf = ksize / 2;
	int width = inputMat->size().width;
	int height = inputMat->size().height;




	Mat* tempReal = new Mat(inputMat->size(), CV_32FC1);
	Mat* tempImag = new Mat(inputMat->size(), CV_32FC1);


	float omegaX, omegaY;
	omegaX = 2 * pi*cos(theta) / lambda;
	omegaY = 2 * pi*sin(theta) / lambda;

	for (int y0 = 0; y0<height; y0++)
		for (int x0 = 0; x0 < width; x0++) {
			tempReal->at<float>(y0, x0) = 0;
			tempImag->at<float>(y0, x0) = 0;
			for (int x = -khalf; x <= khalf; x++) {


				tempReal->at<float>(y0, x0) += atWrapperInAll(inputMat, x0 + x, y0)*expf((-x*x) / (2 * sigma*sigma))*(cos(omegaX*(x)));
				tempImag->at<float>(y0, x0) += atWrapperInAll(inputMat, x0 + x, y0)*expf((-x*x) / (2 * sigma*sigma))*(sin(omegaX*(x)));

			}



		}



	for (int y0 = 0; y0<height; y0++)
		for (int x0 = 0; x0 < width; x0++) {
			resultReal->at<float>(y0, x0) = 0;
			resultImag->at<float>(y0, x0) = 0;
			for (int y = -khalf; y <= khalf; y++) {

				resultReal->at<float>(y0, x0) += atWrapperInAll(tempReal, x0, y0 + y)*expf((-y*y) / (2 * sigma*sigma))*(cos(omegaY*(y))) - atWrapperInAll(tempImag, x0, y0 + y)*expf((-y*y) / (2 * sigma*sigma))*(sin(omegaY*(y)));
				resultImag->at<float>(y0, x0) += atWrapperInAll(tempReal, x0, y0 + y)*expf((-y*y) / (2 * sigma*sigma))*(sin(omegaY*(y))) + atWrapperInAll(tempImag, x0, y0 + y)*expf((-y*y) / (2 * sigma*sigma))*(cos(omegaY*(y)));
			}

			resultReal->at<float>(y0, x0) *= 1 / (2 * pi*sigma*sigma);
			resultImag->at<float>(y0, x0) *= -1 / (2 * pi*sigma*sigma);
		}

	delete tempReal;
	delete tempImag;

}

void fastIIRIsptropicGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag) {
	int khalf = ksize / 2;
	int width = inputMat->size().width;
	int height = inputMat->size().height;

	Mat* w_cos = new Mat(1, width, inputMat->type());
	Mat* w_sin = new Mat(1, width, inputMat->type());
	Mat* tempReal = new Mat(height, width, inputMat->type());
	Mat* tempImag = new Mat(height, width, inputMat->type());
	Mat* w_cosInY = new Mat(height, 1, inputMat->type());
	Mat* w_sinInY = new Mat(height, 1, inputMat->type());



	float m0 = 1.16680, m1 = 1.10783, m2 = 1.40586;
	float q = (sigma <3.556) ? -0.2568 + 0.5784*sigma + 0.0561*sigma*sigma : 2.5091 + 0.9804*(sigma - 3.556);

	float scale = (m0 + q)*(m1*m1 + m2*m2 + 2 * m1*q + q*q);
	float  b1 = -q*(2 * m0*m1 + m1*m1 + m2*m2 + (2 * m0 + 4 * m1)*q + 3 * q*q) / (scale), b2 = q*q*(m0 + 2 * m1 + 3 * q) / scale, b3 = -q*q*q / scale, largeB = m0*(m1*m1 + m2*m2) / scale;
	largeB = largeB*largeB;
	float omegaX = 2 * pi*cos(theta) / lambda;
	float omegaY = 2 * pi*sin(theta) / lambda;


	Mat* cosSig = makeSignalCosineInXYDirection(inputMat, omegaX, omegaY);
	Mat* sinSig = makeSignalSineInXYDirection(inputMat, omegaX, omegaY);
	for (int y = 0; y < height; y++)
	{
		for (int x = 0; x < width; x++)//forwarding
		{
			w_cos->at<float>(0, x) = atWrapperInX(cosSig, x, y)
				- (b1*atWrapperInX(w_cos, x - 1, 0)
				+ b2*atWrapperInX(w_cos, x - 2, 0)
				+ b3*atWrapperInX(w_cos, x - 3, 0));

		}

		for (int x = width - 1; x >= 0; x--)//backwarding
		{
			tempReal->at<float>(y, x) = largeB*atWrapperInX(w_cos, x, 0)
				- (b1*atWrapperInX(tempReal, x + 1, y)
				+ b2*atWrapperInX(tempReal, x + 2, y)
				+ b3*atWrapperInX(tempReal, x + 3, y));
		}


		for (int x = 0; x < width; x++)//forwarding//
		{
			w_sin->at<float>(0, x) = atWrapperInX(sinSig, x, y)
				- (b1*atWrapperInX(w_sin, x - 1, 0)
				+ b2*atWrapperInX(w_sin, x - 2, 0)
				+ b3*atWrapperInX(w_sin, x - 3, 0));

		}

		for (int x = width - 1; x >= 0; x--)//backwarding
		{
			tempImag->at<float>(y, x) = largeB*atWrapperInX(w_sin, x, 0)
				- (b1*atWrapperInX(tempImag, x + 1, y)
				+ b2*atWrapperInX(tempImag, x + 2, y)
				+ b3*atWrapperInX(tempImag, x + 3, y));
		}
	}
	for (int x = 0; x < width; x++)
	{
		for (int y = 0; y < height; y++)
		{
			w_cosInY->at<float>(y, 0) = atWrapperInY(tempReal, x, y) -
				(b1*atWrapperInY(w_cosInY, 0, y - 1) +
				b2*atWrapperInY(w_cosInY, 0, y - 2) +
				b3*atWrapperInY(w_cosInY, 0, y - 3));

		}

		for (int y = height - 1; y >= 0; y--)
		{

			resultReal->at<float>(y, x) = largeB*atWrapperInY(w_cosInY, 0, y) -
				(b1*atWrapperInY(resultReal, x, y + 1) +
				b2*atWrapperInY(resultReal, x, y + 2) +
				b3*atWrapperInY(resultReal, x, y + 3));

		}

		for (int y = 0; y < height; y++)
		{
			w_sinInY->at<float>(y, 0) = atWrapperInY(tempImag, x, y) -
				(b1*atWrapperInY(w_sinInY, 0, y - 1) +
				b2*atWrapperInY(w_sinInY, 0, y - 2) +
				b3*atWrapperInY(w_sinInY, 0, y - 3));

		}

		for (int y = height - 1; y >= 0; y--)
		{

			resultImag->at<float>(y, x) = largeB*atWrapperInY(w_sinInY, 0, y) -
				(b1*atWrapperInY(resultImag, x, y + 1) +
				b2*atWrapperInY(resultImag, x, y + 2) +
				b3*atWrapperInY(resultImag, x, y + 3));

		}

	}


	//demodulation
	for (int x = 0; x< width; x++) {
		for (int y = 0; y<height; y++) {
			float real = resultReal->at<float>(y, x);
			float imag = resultImag->at<float>(y, x);
			resultReal->at<float>(y, x) = cos(omegaX*(-x) + omegaY*(-y))*real - sin(omegaX*(-x) + omegaY*(-y))*imag;
			resultImag->at<float>(y, x) = -(cos(omegaX*(-x) + omegaY*(-y))*imag + sin(omegaX*(-x) + omegaY*(-y))*real);
		}
	}
	delete w_cos;
	delete w_sin;
	delete tempReal;
	delete tempImag;
	delete w_cosInY;
	delete w_sinInY;
	delete cosSig;
	delete sinSig;
}

void recursiveGabor(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag) {
	int khalf = ksize / 2;
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	float u0, v0;

	float m0 = 1.16680, m1 = 1.10783, m2 = 1.40586;
	float q = (sigma <3.556) ? -0.2568 + 0.5784*sigma + 0.0561*sigma*sigma : 2.5091 + 0.9804*(sigma - 3.556);

	float scale = (m0 + q)*(m1*m1 + m2*m2 + 2 * m1*q + q*q);
	float  b1 = -q*(2 * m0*m1 + m1*m1 + m2*m2 + (2 * m0 + 4 * m1)*q + 3 * q*q) / (scale), b2 = q*q*(m0 + 2 * m1 + 3 * q) / scale, b3 = -q*q*q / scale, largeB = m0*(m1*m1 + m2*m2) / scale;
	largeB = largeB*largeB;



	float** imageData = memAllocFloat2(width, height);
	for (int x = 0; x < width; x++)
		for (int y = 0; y < height; y++)
			imageData[x][y] = inputMat->at<float>(y, x);

	float* W_Real_InX = new float[width];
	float* W_Imagin_InX = new float[width];
	float** Out_Real = memAllocFloat2(height, width);
	float** Out_Imagin = memAllocFloat2(height, width);

	float* W_Real_InY = new float[height];
	float* W_Imagin_InY = new float[height];

	float** Result_Real = memAllocFloat2(width, height);
	float** Result_Imagin = memAllocFloat2(width, height);

	float omegaX, omegaY;
	omegaX = 2 * pi*cos(theta) / lambda;
	omegaY = 2 * pi*sin(theta) / lambda;

	float bc1, bs1, bc2, bs2, bc3, bs3;





	bc1 = b1*cos(omegaX); bs1 = b1*sin(omegaX);
	bc2 = b2*cos(omegaX * 2); bs2 = b2*sin(omegaX * 2);
	bc3 = b3*cos(omegaX * 3); bs3 = b3*sin(omegaX * 3);

	for (int y = 0; y < height; y++)
	{
		/*
		Before compute w_cos, we have to exception processing for signal[0,1,2]o
		For solving this problem, see reculsive gaussian
		*/
		float* W_Real_InX_0 = W_Real_InX, *W_Real_InX_1 = W_Real_InX, *W_Real_InX_2 = W_Real_InX, *W_Real_InX_3 = W_Real_InX;
		float* W_Imagin_InX_0 = W_Imagin_InX, *W_Imagin_InX_1 = W_Imagin_InX, *W_Imagin_InX_2 = W_Imagin_InX, *W_Imagin_InX_3 = W_Imagin_InX;
		////////////////////////// boundary exception//////////////////////////////////
		*W_Real_InX_0++ = imageData[0][y];
		*W_Imagin_InX_0++ = 0;

		*W_Real_InX_0++ = imageData[1][y] - (bc1*(*W_Real_InX_1) - bs1*(*W_Imagin_InX_1));
		*W_Imagin_InX_0++ = -(bc1*(*W_Imagin_InX_1++) + bs1*(*W_Real_InX_1++));

		*W_Real_InX_0++ = imageData[2][y] - (bc1*(*W_Real_InX_1) - bs1*(*W_Imagin_InX_1) +
			bc2*(*W_Real_InX_2) - bs2*(*W_Imagin_InX_2));
		*W_Imagin_InX_0++ = -(bc1*(*W_Imagin_InX_1++) + bs1*(*W_Real_InX_1++) +
			bc2*(*W_Imagin_InX_2++) + bs2*(*W_Real_InX_2++));

		////////////////////////// boundary exception//////////////////////////////////

		for (int x = 3; x < width; x++)//forwarding
		{

			*W_Real_InX_0++ = imageData[x][y] - (bc1*(*W_Real_InX_1) - bs1*(*W_Imagin_InX_1) +
				bc2*(*W_Real_InX_2) - bs2*(*W_Imagin_InX_2) +
				bc3*(*W_Real_InX_3) - bs3*(*W_Imagin_InX_3));

			*W_Imagin_InX_0++ = -(bc1*(*W_Imagin_InX_1++) + bs1*(*W_Real_InX_1++) +
				bc2*(*W_Imagin_InX_2++) + bs2*(*W_Real_InX_2++) +
				bc3*(*W_Imagin_InX_3++) + bs3*(*W_Real_InX_3++));

		}

		W_Real_InX_0 = W_Real_InX + width - 1;
		W_Imagin_InX_0 = W_Imagin_InX + width - 1;

		float* Out_Real_0 = Out_Real[y] + width - 1, *Out_Real_1 = Out_Real[y] + width - 1, *Out_Real_2 = Out_Real[y] + width - 1, *Out_Real_3 = Out_Real[y] + width - 1;
		float* Out_Imagin_0 = Out_Imagin[y] + width - 1, *Out_Imagin_1 = Out_Imagin[y] + width - 1, *Out_Imagin_2 = Out_Imagin[y] + width - 1, *Out_Imagin_3 = Out_Imagin[y] + width - 1;

		////////////////////////// boundary exception//////////////////////////////////
		*Out_Real_0-- = largeB*(*W_Real_InX_0--);
		*Out_Imagin_0-- = largeB*(*W_Imagin_InX_0--);

		*Out_Real_0-- = largeB*(*W_Real_InX_0--) - (bc1*(*Out_Real_1) + bs1*(*Out_Imagin_1));
		*Out_Imagin_0-- = largeB*(*W_Imagin_InX_0--) - (bc1*(*Out_Imagin_1--) - bs1*(*Out_Real_1--));

		*Out_Real_0-- = largeB*(*W_Real_InX_0--) - (bc1*(*Out_Real_1) + bs1*(*Out_Imagin_1) +
			bc2*(*Out_Real_2) + bs2*(*Out_Imagin_2));
		*Out_Imagin_0-- = largeB*(*W_Imagin_InX_0--) - (bc1*(*Out_Imagin_1--) - bs1*(*Out_Real_1--) +
			bc2*(*Out_Imagin_2--) - bs2*(*Out_Real_2--));

		////////////////////////// boundary exception//////////////////////////////////


		for (int x = width - 1 - 3; x >= 0; x--)//backwarding
		{					 //transpose  


			*Out_Real_0-- = largeB * (*W_Real_InX_0--) - (bc1*(*Out_Real_1) + bs1*(*Out_Imagin_1) +
				bc2*(*Out_Real_2) + bs2*(*Out_Imagin_2) +
				bc3*(*Out_Real_3) + bs3*(*Out_Imagin_3));
			*Out_Imagin_0-- = largeB * (*W_Imagin_InX_0--) - (bc1*(*Out_Imagin_1--) - bs1*(*Out_Real_1--) +
				bc2*(*Out_Imagin_2--) - bs2*(*Out_Real_2--) +
				bc3*(*Out_Imagin_3--) - bs3*(*Out_Real_3--));
		}

	}



	bc1 = b1*cos(omegaY); bs1 = b1*sin(omegaY);
	bc2 = b2*cos(omegaY * 2); bs2 = b2*sin(omegaY * 2);
	bc3 = b3*cos(omegaY * 3); bs3 = b3*sin(omegaY * 3);




	for (int x = 0; x < width; x++)
	{

		float* W_Real_InY_0 = W_Real_InY, *W_Real_InY_1 = W_Real_InY, *W_Real_InY_2 = W_Real_InY, *W_Real_InY_3 = W_Real_InY;
		float* W_Imagin_InY_0 = W_Imagin_InY, *W_Imagin_InY_1 = W_Imagin_InY, *W_Imagin_InY_2 = W_Imagin_InY, *W_Imagin_InY_3 = W_Imagin_InY;


		////////////////////////// boundary exception//////////////////////////////////
		*W_Real_InY_0++ = Out_Real[0][x];
		*W_Imagin_InY_0++ = Out_Imagin[0][x];

		*W_Real_InY_0++ = Out_Real[1][x] - (bc1*(*W_Real_InY_1) - bs1*(*W_Imagin_InY_1));
		*W_Imagin_InY_0++ = Out_Imagin[1][x] - (bc1*(*W_Imagin_InY_1++) + bs1*(*W_Real_InY_1++));

		*W_Real_InY_0++ = Out_Real[2][x] - (bc1*(*W_Real_InY_1) - bs1*(*W_Imagin_InY_1) +
			bc2*(*W_Real_InY_2) - bs2*(*W_Imagin_InY_2));
		*W_Imagin_InY_0++ = Out_Imagin[2][x] - (bc1*(*W_Imagin_InY_1++) + bs1*(*W_Real_InY_1++) +
			bc2*(*W_Imagin_InY_2++) + bs2*(*W_Real_InY_2++));

		////////////////////////// boundary exception//////////////////////////////////


		for (int y = 3; y < height; y++)
		{

			*W_Real_InY_0++ = Out_Real[y][x] - (bc1*(*W_Real_InY_1) - bs1*(*W_Imagin_InY_1) +
				bc2*(*W_Real_InY_2) - bs2*(*W_Imagin_InY_2) +
				bc3*(*W_Real_InY_3) - bs3*(*W_Imagin_InY_3));
			*W_Imagin_InY_0++ = Out_Imagin[y][x] - (bc1*(*W_Imagin_InY_1++) + bs1*(*W_Real_InY_1++) +
				bc2*(*W_Imagin_InY_2++) + bs2*(*W_Real_InY_2++) +
				bc3*(*W_Imagin_InY_3++) + bs3*(*W_Real_InY_3++));


		}

		W_Real_InY_0 = W_Real_InY + height - 1;
		W_Imagin_InY_0 = W_Imagin_InY + height - 1;

		float* Result_Real_0 = Result_Real[x] + height - 1, *Result_Real_1 = Result_Real[x] + height - 1, *Result_Real_2 = Result_Real[x] + height - 1, *Result_Real_3 = Result_Real[x] + height - 1;
		float* Result_Imagin_0 = Result_Imagin[x] + height - 1, *Result_Imagin_1 = Result_Imagin[x] + height - 1, *Result_Imagin_2 = Result_Imagin[x] + height - 1, *Result_Imagin_3 = Result_Imagin[x] + height - 1;


		////////////////////////// boundary exception//////////////////////////////////
		*Result_Real_0-- = largeB*(*W_Real_InY_0--);
		*Result_Imagin_0-- = largeB*(*W_Imagin_InY_0--);

		*Result_Real_0-- = largeB*(*W_Real_InY_0--) - (bc1*(*Result_Real_1) + bs1*(*Result_Imagin_1));
		*Result_Imagin_0-- = largeB*(*W_Imagin_InY_0--) - (bc1*(*Result_Imagin_1--) - bs1*(*Result_Real_1--));

		*Result_Real_0-- = largeB*(*W_Real_InY_0--) - (bc1*(*Result_Real_1) + bs1*(*Result_Imagin_1) +
			bc2*(*Result_Real_2) + bs2*(*Result_Imagin_2));
		*Result_Imagin_0-- = largeB*(*W_Imagin_InY_0--) - (bc1*(*Result_Imagin_1--) - bs1*(*Result_Real_1--) +
			bc2*(*Result_Imagin_2--) - bs2*(*Result_Real_2--));

		////////////////////////// boundary exception//////////////////////////////////



		for (int y = height - 1 - 3; y >= 0; y--)
		{

			*Result_Real_0-- = largeB*(*W_Real_InY_0--) - (bc1*(*Result_Real_1) + bs1*(*Result_Imagin_1) +
				bc2*(*Result_Real_2) + bs2*(*Result_Imagin_2) +
				bc3*(*Result_Real_3) + bs3*(*Result_Imagin_3));
			*Result_Imagin_0-- = largeB*(*W_Imagin_InY_0--) - (bc1*(*Result_Imagin_1--) - bs1*(*Result_Real_1--) +
				bc2*(*Result_Imagin_2--) - bs2*(*Result_Real_2--) +
				bc3*(*Result_Imagin_3--) - bs3*(*Result_Real_3--));

		}


	}

	for (int x = 0; x<width; x++)
		for (int y = 0; y < height; y++) {
			resultReal->at<float>(y, x) = Result_Real[x][y];
			resultImag->at<float>(y, x) = Result_Imagin[x][y];
		}




	memFreeFloat1(W_Real_InX);
	memFreeFloat1(W_Imagin_InX);
	memFreeFloat1(W_Real_InY);
	memFreeFloat1(W_Imagin_InY);
	memFreeFloat2(Out_Real);
	memFreeFloat2(Out_Imagin);
	memFreeFloat2(imageData);
	memFreeFloat2(Result_Real);
	memFreeFloat2(Result_Imagin);
}

void ourMethod(Mat* inputMat, int ksize, float sigma, float lambda, float theta, Mat* resultReal, Mat* resultImag, Mat* symmetricReal, Mat* symmetricImag) {
	int khalf = ksize / 2;
	int width = inputMat->size().width;
	int height = inputMat->size().height;

	/*

	Have to compute sigma

	*/

	float m0 = 1.16680, m1 = 1.10783, m2 = 1.40586;
	float q = (sigma <3.556) ? -0.2568 + 0.5784*sigma + 0.0561*sigma*sigma : 2.5091 + 0.9804*(sigma - 3.556);

	float scale = (m0 + q)*(m1*m1 + m2*m2 + 2 * m1*q + q*q);
	float  b1 = -q*(2 * m0*m1 + m1*m1 + m2*m2 + (2 * m0 + 4 * m1)*q + 3 * q*q) / (scale), b2 = q*q*(m0 + 2 * m1 + 3 * q) / scale, b3 = -q*q*q / scale, largeB = m0*(m1*m1 + m2*m2) / scale;
	largeB = largeB*largeB;

	float first, second, third, forth;
	Mat* Out_Real = new Mat(inputMat->size(), inputMat->type());
	Mat* Out_Imagin = new Mat(inputMat->size(), inputMat->type());
	Mat* w_cos = new Mat(1, width, inputMat->type());
	Mat* w_sin = new Mat(1, width, inputMat->type());
	Mat* tempReal = new Mat(height, width, inputMat->type());
	Mat* tempImag = new Mat(height, width, inputMat->type());
	Mat* w_cosReal = new Mat(height, 1, inputMat->type());
	Mat* w_sinReal = new Mat(height, 1, inputMat->type());
	Mat* w_cosImag = new Mat(height, 1, inputMat->type());
	Mat* w_sinImag = new Mat(height, 1, inputMat->type());
	Mat* out_cosReal = new Mat(height, 1, inputMat->type());
	Mat* out_cosImag = new Mat(height, 1, inputMat->type());
	Mat* out_sinImag = new Mat(height, 1, inputMat->type());
	Mat* out_sinReal = new Mat(height, 1, inputMat->type());

	float omegaX, omegaY;
	omegaX = 2 * pi*cos(theta) / lambda;
	omegaY = 2 * pi*sin(theta) / lambda;


	Mat* CosSig = makeSignalCosine(inputMat, omegaX);
	Mat* SineSig = makeSignalSine(inputMat, omegaX);



	for (int y = 0; y < height; y++)
	{
		/*
		Before compute w_cos, we have to exception processing for signal[0,1,2] -> replace by using function
		For solving this problem, see reculsive gaussian
		*/


		for (int x = 0; x < width; x++)//forwarding//
		{
			w_cos->at<float>(0, x) = atWrapperInX(CosSig, x, y) -
				(b1*atWrapperInX(w_cos, x - 1, 0)
				+ b2*atWrapperInX(w_cos, x - 2, 0)
				+ b3*atWrapperInX(w_cos, x - 3, 0));
			w_sin->at<float>(0, x) = (atWrapperInX(SineSig, x, y)) -
				(b1*atWrapperInX(w_sin, x - 1, 0)
				+ b2*atWrapperInX(w_sin, x - 2, 0)
				+ b3*atWrapperInX(w_sin, x - 3, 0));
		}

		for (int x = width - 1; x >= 0; x--)//backwarding
		{
			Out_Real->at<float>(0, x) = largeB*atWrapperInX(w_cos, x, 0)
				- (b1*atWrapperInX(Out_Real, x + 1, 0)
				+ b2*atWrapperInX(Out_Real, x + 2, 0)
				+ b3*atWrapperInX(Out_Real, x + 3, 0));
			Out_Imagin->at<float>(0, x) = largeB*atWrapperInX(w_sin, x, 0)
				- (b1*atWrapperInX(Out_Imagin, x + 1, 0)
				+ b2*atWrapperInX(Out_Imagin, x + 2, 0)
				+ b3*atWrapperInX(Out_Imagin, x + 3, 0));


			tempReal->at<float>(y, x) = { cos(omegaX*(x))*atWrapperInX(Out_Real, x, 0)
				+ sin(omegaX*(x))*atWrapperInX(Out_Imagin, x, 0) };
			tempImag->at<float>(y, x) = { cos(omegaX*(x))*atWrapperInX(Out_Imagin, x, 0)
				- sin(omegaX*(x))*atWrapperInX(Out_Real, x, 0) };
		}
	}



	Mat* cosRealSig = makeSignalCosine2(tempReal, omegaY);
	Mat* sinRealSig = makeSignalSine2(tempReal, omegaY);
	Mat* cosImagSig = makeSignalCosine2(tempImag, omegaY);
	Mat* sinImagSig = makeSignalSine2(tempImag, omegaY);

	for (int x = 0; x < width; x++)
	{
		for (int y = 0; y < height; y++)
		{
			w_cosReal->at<float>(y, 0) = atWrapperInY(cosRealSig, x, y) -
				(b1*atWrapperInY(w_cosReal, 0, y - 1) +
				b2*atWrapperInY(w_cosReal, 0, y - 2) +
				b3*atWrapperInY(w_cosReal, 0, y - 3));
			w_sinReal->at<float>(y, 0) = atWrapperInY(sinRealSig, x, y) -
				(b1*atWrapperInY(w_sinReal, 0, y - 1) +
				b2*atWrapperInY(w_sinReal, 0, y - 2) +
				b3*atWrapperInY(w_sinReal, 0, y - 3));
			w_cosImag->at<float>(y, 0) = atWrapperInY(cosImagSig, x, y) -
				(b1*atWrapperInY(w_cosImag, 0, y - 1) +
				b2*atWrapperInY(w_cosImag, 0, y - 2) +
				b3*atWrapperInY(w_cosImag, 0, y - 3));
			w_sinImag->at<float>(y, 0) = atWrapperInY(sinImagSig, x, y) -
				(b1*atWrapperInY(w_sinImag, 0, y - 1) +
				b2*atWrapperInY(w_sinImag, 0, y - 2) +
				b3*atWrapperInY(w_sinImag, 0, y - 3));
		}

		for (int y = height - 1; y >= 0; y--)
		{

			out_cosReal->at<float>(y, 0) = largeB*atWrapperInY(w_cosReal, 0, y) -
				(b1*atWrapperInY(out_cosReal, 0, y + 1) +
				b2*atWrapperInY(out_cosReal, 0, y + 2) +
				b3*atWrapperInY(out_cosReal, 0, y + 3));
			out_sinReal->at<float>(y, 0) = largeB*atWrapperInY(w_sinReal, 0, y) -
				(b1*atWrapperInY(out_sinReal, 0, y + 1) +
				b2*atWrapperInY(out_sinReal, 0, y + 2) +
				b3*atWrapperInY(out_sinReal, 0, y + 3));
			out_cosImag->at<float>(y, 0) = largeB*atWrapperInY(w_cosImag, 0, y) -
				(b1*atWrapperInY(out_cosImag, 0, y + 1) +
				b2*atWrapperInY(out_cosImag, 0, y + 2) +
				b3*atWrapperInY(out_cosImag, 0, y + 3));
			out_sinImag->at<float>(y, 0) = largeB*atWrapperInY(w_sinImag, 0, y) -
				(b1*atWrapperInY(out_sinImag, 0, y + 1) +
				b2*atWrapperInY(out_sinImag, 0, y + 2) +
				b3*atWrapperInY(out_sinImag, 0, y + 3));

			first = cos(omegaY*(y))*atWrapperInY(out_cosReal, 0, y) +
				sin(omegaY*(y))*atWrapperInY(out_sinReal, 0, y);
			second = cos(omegaY*(y))*atWrapperInY(out_sinImag, 0, y) -
				sin(omegaY*(y))*atWrapperInY(out_cosImag, 0, y);
			third = cos(omegaY*(y))*atWrapperInY(out_sinReal, 0, y) -
				sin(omegaY*(y))*atWrapperInY(out_cosReal, 0, y);
			forth = cos(omegaY*(y))*atWrapperInY(out_cosImag, 0, y) +
				sin(omegaY*(y))*atWrapperInY(out_sinImag, 0, y);

			resultReal->at<float>(y, x) = first - second;
			resultImag->at<float>(y, x) = -(third + forth);




		}
	}
	delete Out_Real;
	delete Out_Imagin;
	delete w_cos;
	delete w_sin;
	delete tempReal;
	delete tempImag;
	delete w_cosReal;
	delete w_sinReal;
	delete w_cosImag;
	delete w_sinImag;
	delete out_cosReal;
	delete out_cosImag;
	delete out_sinImag;
	delete out_sinReal;
	delete CosSig;
	delete SineSig;
	delete cosRealSig;
	delete sinRealSig;
	delete cosImagSig;
	delete sinImagSig;
}



void ourImwrite(Mat inputR, Mat inputI, string n, int u, int v) {
	Mat test = Mat::zeros(inputR.size(), CV_32FC1);


	Mat Normtest;






	for (int y = 0; y < test.size().height; y++)
		for (int x = 0; x < test.size().width; x++) {
			test.at<float>(y, x) = 255 * sqrt(inputR.at<float>(y, x)*inputR.at<float>(y, x) + inputI.at<float>(y, x)*inputI.at<float>(y, x));

		}

	cv::normalize(test, Normtest, 0, 1, cv::NORM_MINMAX);
	Normtest = Normtest * 255;
	Mat mat2;
	Normtest.convertTo(mat2, CV_8U);


	string name = "images/";


	name.append(string("u = "));
	name.append(to_string(u));
	name.append(string("  v = "));
	name.append(to_string(v));
	name.append(n);
	name.append(string(".jpg"));



	imwrite(name, mat2);

}

void ourImshow(Mat inputR, Mat inputI, int u, int v) {
	Mat test = Mat::zeros(inputR.size(), CV_32FC1);


	Mat Normtest;






	for (int y = 0; y < test.size().height; y++)
		for (int x = 0; x < test.size().width; x++) {
			test.at<float>(y, x) = 255 * sqrt(inputR.at<float>(y, x)*inputR.at<float>(y, x) + inputI.at<float>(y, x)*inputI.at<float>(y, x));

		}

	cv::normalize(test, Normtest, 0, 1, cv::NORM_MINMAX);

	string name = "image u = ";
	name.append(to_string(u));
	name.append(string("  v = "));
	name.append(to_string(v));


	imshow(name, Normtest);

}




Mat* makeSignalCosineInXYDirection(Mat* inputMat, float omegaX, float omegaY) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int x = 0; x < width; x++) {
		for (int y = 0; y < height; y++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*cos(omegaX*x + omegaY*y);
		}
	}
	return ret;
}
Mat* makeSignalSineInXYDirection(Mat* inputMat, float omegaX, float omegaY) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int x = 0; x < width; x++) {
		for (int y = 0; y < height; y++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*sin(omegaX*x + omegaY*y);
		}
	}
	return ret;
}

float atWrapperInX(Mat* m, int x, int y) {

	if (x >= m->size().width || x < 0) {
		return 0;
	}

	return m->at<float>(y, x);

}
float atWrapperInY(Mat* m, int x, int y) {

	if (y >= m->size().height || y < 0) {
		return 0;
	}

	return m->at<float>(y, x);

}
float atWrapperInAll(Mat* m, int x, int y) {

	if (y >= m->size().height || y < 0 || x >= m->size().width || x < 0) {
		return 0;
	}

	return m->at<float>(y, x);

}



Mat* makeSignalCosine(Mat* inputMat, float omegaX) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int x = 0; x < width; x++) {
		float c = (float)cos(omegaX*x);
		for (int y = 0; y < height; y++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*c;
		}
	}
	return ret;
}
Mat* makeSignalSine(Mat* inputMat, float omegaX) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int x = 0; x < width; x++) {
		float c = (float)sin(omegaX*x);
		for (int y = 0; y < height; y++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*c;
		}
	}
	return ret;
}
Mat* makeSignalCosine2(Mat* inputMat, float omegaY) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int y = 0; y < height; y++)
	{
		float c = (float)cos(omegaY*y);
		for (int x = 0; x < width; x++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*c;
		}
	}
	return ret;
}
Mat* makeSignalSine2(Mat* inputMat, float omegaY) {
	int width = inputMat->size().width;
	int height = inputMat->size().height;
	Mat* ret = new Mat(height, width, inputMat->type());
	for (int y = 0; y < height; y++)
	{
		float c = (float)sin(omegaY*y);
		for (int x = 0; x < width; x++) {
			ret->at<float>(y, x) = inputMat->at<float>(y, x)*c;
		}
	}
	return ret;
}

float* memAllocFloat1(int x)
{
	int padding = 10;
	float *a;
	a = (float*)malloc(sizeof(float)*(x));
	//	if (a == NULL) { print("mem is too huge."); }
	return(a);
}
float *** memAllocFloat3(int x, int y, int z) {
	int padding = 10;
	float *a, **p, ***pp;
	int yz = y*z;
	int i, j;
	a = (float*)malloc(sizeof(float)*(x*yz));
	if (a == NULL) { printf("mem is too huge."); }
	p = (float**)malloc(sizeof(float*)*x*y);
	pp = (float***)malloc(sizeof(float**)*x);
	for (i = 0; i<x; i++)
		for (j = 0; j<y; j++)
			p[i*y + j] = &a[i*yz + j*z];
	for (i = 0; i<x; i++)
		pp[i] = &p[i*y];
	return(pp);
}

void memFreeFloat3(float ***p)
{
	if (p != NULL)
	{
		free(p[0][0]);
		free(p[0]);
		free(p);
		p = NULL;
	}
}

float** memAllocFloat2(int x, int y)
{
	int padding = 10;
	float *a, **p;
	a = (float*)malloc(sizeof(float)*(x*y));
	if (a == NULL) { printf("mem is too huge."); }
	p = (float**)malloc(sizeof(float*)*x);
	for (int i = 0; i<x; i++) p[i] = &a[i*y];
	return(p);
}

void memFreeFloat1(float *p)
{
	if (p != NULL)
	{
		free(p);
		p = NULL;
	}
}
void memFreeFloat2(float **p)
{
	if (p != NULL)
	{
		free(p[0]);
		free(p);
		p = NULL;
	}
}
float**** memAllocFloat4(int u, int v, int x, int y) {
	int padding = 10;
	float *a, **p, ***pp, ****ppp;
	int xy = x*y;
	int uv = u*v;
	int vx = v*x;
	int i, j, k;
	a = (float*)malloc(sizeof(float)*(uv*xy));
	if (a == NULL) { printf("mem is too huge."); }
	p = (float**)malloc(sizeof(float*)*uv*x);
	pp = (float***)malloc(sizeof(float**)*uv);
	ppp = (float****)malloc(sizeof(float***)*u);
	for (i = 0; i < u; i++)
		for (j = 0; j < v; j++)
			for (k = 0; k < x; k++)
				p[i*vx + j*x + k] = &a[i*v*xy + j*xy + k*y];
	for (i = 0; i < u; i++)
		for (j = 0; j < v; j++)
			pp[i*v + j] = &p[i*v*x + j*x];
	for (i = 0; i < u; i++)
		ppp[i] = &pp[i*v];
	return(ppp);

}
void memFreeFloat4(float ****p)
{
	if (p != NULL)
	{
		free(p[0][0][0]);
		free(p[0][0]);
		free(p[0]);
		free(p);
		p = NULL;
	}
}

